<?php
// auth/auth.php
declare(strict_types=1);

require_once __DIR__ . '/../config/db.php';

function start_session_once(): void
{
    if (session_status() !== PHP_SESSION_ACTIVE) {
        session_set_cookie_params([
            'httponly' => true,
            'samesite' => 'Lax',
        ]);
        session_start();
    }
}

if (!function_exists('h')) {
    function h(string $v): string {
        return htmlspecialchars($v, ENT_QUOTES, 'UTF-8');
    }
}

function is_logged_in(): bool
{
    start_session_once();
    return !empty($_SESSION['admin_id']);
}

function require_login(): void
{
    if (!is_logged_in()) {
        header('Location: /couture_site/auth/login.php');
        exit;
    }
}

function admin_username(): string
{
    start_session_once();
    return (string)($_SESSION['admin_username'] ?? '');
}

function login_admin(string $username, string $password): bool
{
    start_session_once();

    $username = trim($username);
    if ($username === '' || $password === '') {
        return false;
    }

    $pdo = db();
    $stmt = $pdo->prepare("SELECT id, username, password_hash FROM admins WHERE username = :u LIMIT 1");
    $stmt->execute([':u' => $username]);
    $row = $stmt->fetch();

    if (!$row) {
        return false;
    }

    if (!password_verify($password, (string)$row['password_hash'])) {
        return false;
    }

    session_regenerate_id(true);

    $_SESSION['admin_id'] = (int)$row['id'];
    $_SESSION['admin_username'] = (string)$row['username'];

    return true;
}

function logout_admin(): void
{
    start_session_once();
    $_SESSION = [];

    if (ini_get('session.use_cookies')) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000, $params['path'] ?? '/', $params['domain'] ?? '', false, true);
    }

    session_destroy();
}
