<?php
// admin/gallery_form.php
declare(strict_types=1);

require_once __DIR__ . '/../auth/auth.php';
require_login();

require_once __DIR__ . '/../config/db.php';
$pdo = db();

$cfg = require __DIR__ . '/../config/config.php';
$galleryUploadDir = $cfg['gallery_upload_dir']; // chemin serveur
$maxUploadMb = (int)($cfg['max_upload_mb'] ?? 8);

function h(string $v): string {
    return htmlspecialchars($v, ENT_QUOTES, 'UTF-8');
}

function ensure_dir(string $path): void {
    if (!is_dir($path)) {
        @mkdir($path, 0777, true);
    }
}

function is_allowed_image_mime(string $mime): bool {
    $allowed = ['image/jpeg', 'image/png', 'image/webp'];
    return in_array($mime, $allowed, true);
}

function safe_filename(string $original): string {
    $original = preg_replace('/[^\w\-.]+/u', '-', $original);
    $original = trim((string)$original, '-');
    if ($original === '') $original = 'image';
    return $original;
}

ensure_dir($galleryUploadDir);

$id = (int)($_GET['id'] ?? 0);
$isEdit = $id > 0;

$title = '';
$description = '';
$category = '';
$is_published = 1;
$currentImagePath = '';

$err = '';
$ok = '';

if ($isEdit) {
    $stmt = $pdo->prepare("SELECT * FROM gallery_items WHERE id = :id");
    $stmt->execute([':id' => $id]);
    $row = $stmt->fetch();
    if (!$row) {
        header('Location: /couture_site/admin/gallery_list.php');
        exit;
    }
    $title = (string)$row['title'];
    $description = (string)($row['description'] ?? '');
    $category = (string)($row['category'] ?? '');
    $is_published = (int)$row['is_published'];
    $currentImagePath = (string)$row['image_path'];
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = trim($_POST['title'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $category = trim($_POST['category'] ?? '');
    $is_published = isset($_POST['is_published']) ? 1 : 0;

    if ($title === '') {
        $err = "Le titre est obligatoire.";
    } else {
        // Gestion upload optionnel
        $newImageRelativePath = '';

        if (isset($_FILES['image']) && is_array($_FILES['image']) && ($_FILES['image']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_NO_FILE) {
            $fileErr = (int)($_FILES['image']['error'] ?? UPLOAD_ERR_NO_FILE);

            if ($fileErr !== UPLOAD_ERR_OK) {
                $err = "Erreur upload (code $fileErr).";
            } else {
                $tmp = (string)$_FILES['image']['tmp_name'];
                $size = (int)($_FILES['image']['size'] ?? 0);
                $origName = (string)($_FILES['image']['name'] ?? 'image');

                $maxBytes = $maxUploadMb * 1024 * 1024;
                if ($size <= 0 || $size > $maxBytes) {
                    $err = "Image trop grosse (max {$maxUploadMb} MB).";
                } else {
                    $finfo = new finfo(FILEINFO_MIME_TYPE);
                    $mime = $finfo->file($tmp) ?: '';
                    if (!is_allowed_image_mime($mime)) {
                        $err = "Format image non autorisé (JPEG/PNG/WebP seulement).";
                    } else {
                        // Extension selon mime
                        $ext = match ($mime) {
                            'image/jpeg' => 'jpg',
                            'image/png'  => 'png',
                            'image/webp' => 'webp',
                            default      => 'img',
                        };

                        $base = pathinfo($origName, PATHINFO_FILENAME);
                        $base = safe_filename($base);

                        $random = bin2hex(random_bytes(8));
                        $filename = $base . '-' . $random . '.' . $ext;

                        $destAbs = rtrim($galleryUploadDir, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $filename;
                        if (!move_uploaded_file($tmp, $destAbs)) {
                            $err = "Impossible de déplacer le fichier uploadé.";
                        } else {
                            // Chemin relatif stocké en DB (ex: uploads/gallery/xxx.jpg)
                            $newImageRelativePath = 'uploads/gallery/' . $filename;
                        }
                    }
                }
            }
        }

        if ($err === '') {
            try {
                if ($isEdit) {
                    // si nouvelle image -> supprimer l’ancienne
                    if ($newImageRelativePath !== '' && $currentImagePath !== '') {
                        $oldAbs = __DIR__ . '/../' . $currentImagePath;
                        if (is_file($oldAbs)) {
                            @unlink($oldAbs);
                        }
                        $currentImagePath = $newImageRelativePath;
                    }

                    // si pas d’image et pas d’ancienne -> erreur (car NOT NULL en DB)
                    if ($currentImagePath === '') {
                        $err = "Une image est obligatoire.";
                    } else {
                        $stmt = $pdo->prepare("
                            UPDATE gallery_items
                            SET title = :t, description = :d, category = :c, image_path = :p, is_published = :pub, updated_at = NOW()
                            WHERE id = :id
                        ");
                        $stmt->execute([
                            ':t' => $title,
                            ':d' => ($description !== '' ? $description : null),
                            ':c' => ($category !== '' ? $category : null),
                            ':p' => $currentImagePath,
                            ':pub' => $is_published,
                            ':id' => $id,
                        ]);
                        header('Location: /couture_site/admin/gallery_list.php');
                        exit;
                    }
                } else {
                    // ajout: image obligatoire
                    if ($newImageRelativePath === '') {
                        $err = "Une image est obligatoire (ajout).";
                    } else {
                        $stmt = $pdo->prepare("
                            INSERT INTO gallery_items (title, description, category, image_path, is_published)
                            VALUES (:t, :d, :c, :p, :pub)
                        ");
                        $stmt->execute([
                            ':t' => $title,
                            ':d' => ($description !== '' ? $description : null),
                            ':c' => ($category !== '' ? $category : null),
                            ':p' => $newImageRelativePath,
                            ':pub' => $is_published,
                        ]);
                        header('Location: /couture_site/admin/gallery_list.php');
                        exit;
                    }
                }
            } catch (Throwable $e) {
                $err = "Erreur serveur lors de l’enregistrement.";
            }
        }
    }
}
?>
<!doctype html>
<html lang="fr">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Admin — Galerie <?= $isEdit ? 'Modifier' : 'Ajouter' ?></title>
  <link rel="stylesheet" href="/couture_site/public/assets/css/style.css">
</head>
<body>
<header class="site-header">
  <div class="container header-row">
    <div class="brand"><a href="/couture_site/admin/dashboard.php">Admin</a></div>
    <nav class="nav">
      <a href="/couture_site/admin/dashboard.php">Dashboard</a>
      <a class="active" href="/couture_site/admin/gallery_list.php">Galerie</a>
      <a href="/couture_site/public/index.php">Site</a>
      <a href="/couture_site/auth/logout.php">Déconnexion</a>
    </nav>
  </div>
</header>

<main class="container">
  <div class="card">
    <h1><?= $isEdit ? 'Modifier' : 'Ajouter' ?> une réalisation</h1>

    <?php if ($err): ?><div class="notice err"><?= h($err) ?></div><?php endif; ?>

    <form method="post" action="" enctype="multipart/form-data">
      <label>Titre *</label>
      <input name="title" value="<?= h($title) ?>" required>

      <label>Catégorie</label>
      <input name="category" value="<?= h($category) ?>" placeholder="ex: Robe, Retouche, Sur mesure">

      <label>Description</label>
      <textarea name="description"><?= h($description) ?></textarea>

      <label>Image <?= $isEdit ? '(optionnel si tu gardes l’ancienne)' : '*' ?></label>
      <input type="file" name="image" accept=".jpg,.jpeg,.png,.webp,image/jpeg,image/png,image/webp">

      <?php if ($isEdit && $currentImagePath !== ''): ?>
        <p style="margin-top:10px; margin-bottom:0;"><strong>Image actuelle :</strong></p>
        <img
          src="/couture_site/<?= h($currentImagePath) ?>"
          alt=""
          style="max-width:320px; width:100%; height:auto; border-radius:12px; border:1px solid #eaeaea; margin-top:8px;"
        >
      <?php endif; ?>

      <label style="margin-top:12px; display:flex; gap:10px; align-items:center;">
        <input type="checkbox" name="is_published" <?= ($is_published === 1) ? 'checked' : '' ?> style="width:auto;">
        Publier
      </label>

      <p style="margin-top:12px; display:flex; gap:10px; flex-wrap:wrap;">
        <button class="btn" type="submit">Enregistrer</button>
        <a class="btn outline" href="/couture_site/admin/gallery_list.php">Annuler</a>
      </p>

      <p><small>Formats: JPEG/PNG/WebP. Taille max: <?= (int)$maxUploadMb ?> MB.</small></p>
    </form>
  </div>
</main>
</body>
</html>
